/*
 * Decompiled with CFR 0.152.
 */
package ghidra.file.formats.cart;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import ghidra.app.util.bin.BinaryReader;
import ghidra.app.util.bin.ByteProvider;
import ghidra.file.formats.cart.CartCancelDialogs;
import ghidra.file.formats.cart.CartConfigurationException;
import ghidra.file.formats.cart.CartInvalidARC4KeyException;
import ghidra.file.formats.cart.CartInvalidCartException;
import ghidra.file.formats.cart.CartV1Constants;
import ghidra.file.formats.cart.CartV1Decryptor;
import ghidra.file.formats.cart.CartV1Footer;
import ghidra.file.formats.cart.CartV1Header;
import ghidra.file.formats.cart.CartV1PayloadExtractor;
import ghidra.util.exception.CancelledException;
import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.HexFormat;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.text.StringEscapeUtils;

public class CartV1File {
    private static short version = 1;
    private String name = "<Unknown>";
    private String path = "<Unknown>";
    private long dataOffset = -1L;
    private long payloadOriginalSize = -1L;
    private long packedSize = -1L;
    private Map<String, byte[]> footerHashes;
    private CartV1Header header;
    private CartV1Footer footer;
    private CartV1Decryptor decryptor;
    private long readerLength = -1L;
    private BinaryReader internalReader;

    public CartV1File(ByteProvider byteProvider) throws IOException, CartInvalidCartException, CartInvalidARC4KeyException, CartConfigurationException, CancelledException {
        this(new BinaryReader(byteProvider, true));
    }

    public CartV1File(ByteProvider byteProvider, String arc4Key) throws IOException, CartInvalidCartException, CartInvalidARC4KeyException, CartConfigurationException, CancelledException {
        this(new BinaryReader(byteProvider, true), arc4Key);
    }

    public CartV1File(BinaryReader reader) throws IOException, CartInvalidCartException, CartInvalidARC4KeyException, CartConfigurationException, CancelledException {
        this(reader, null);
    }

    public CartV1File(BinaryReader reader, String arc4Key) throws IOException, CartInvalidCartException, CartInvalidARC4KeyException, CartConfigurationException, CancelledException {
        if (!reader.isLittleEndian()) {
            throw new IOException("CaRT BinaryReader must be Little-Endian.");
        }
        this.readerLength = reader.length();
        if (this.readerLength < 66L) {
            throw new CartInvalidCartException("Data too small to be CaRT format.");
        }
        this.internalReader = reader.clone(0L);
        this.header = new CartV1Header(this.internalReader);
        this.footer = new CartV1Footer(this.internalReader);
        this.name = this.internalReader.getByteProvider().getName();
        this.dataOffset = this.header.dataStart();
        this.packedSize = this.footer.optionalFooterPosition() - this.dataOffset;
        if (this.packedSize <= 0L || this.packedSize > this.readerLength) {
            throw new CartInvalidCartException("Error calculating CaRT compressed payload size.");
        }
        if (arc4Key == null) {
            this.createDecryptor();
        } else {
            this.createDecryptor(arc4Key);
        }
        try {
            this.header.loadOptionalHeader(this.decryptor);
        }
        catch (CartInvalidARC4KeyException e) {
            throw new CartInvalidARC4KeyException("Decryption failed for header metadata: " + String.valueOf(e));
        }
        try {
            this.footer.loadOptionalFooter(this.decryptor);
        }
        catch (CartInvalidARC4KeyException e) {
            throw new CartInvalidARC4KeyException("Decryption failed for footer metadata: " + String.valueOf(e));
        }
        JsonObject optionalHeaderData = this.header.optionalHeaderData();
        if (optionalHeaderData.has("name")) {
            this.path = optionalHeaderData.get("name").getAsString();
        } else {
            this.path = this.name;
            this.path = this.path.endsWith(".cart") ? this.path.substring(0, this.path.length() - 5) : this.path + ".uncart";
        }
        JsonObject optionalFooterData = this.footer.optionalFooterData();
        if (optionalFooterData.has("length")) {
            this.payloadOriginalSize = optionalFooterData.get("length").getAsLong();
            if (this.payloadOriginalSize < 0L) {
                throw new CartInvalidCartException("Bad payload length in footer.");
            }
            if (this.payloadOriginalSize > this.packedSize * 10L && !CartCancelDialogs.promptWarningContinue("Size Warning", "CaRT footer reports payload size <b>" + StringEscapeUtils.escapeHtml4((String)String.valueOf(this.payloadOriginalSize)) + "</b>, but this value seems unreasonable given the compressed size of <i>" + StringEscapeUtils.escapeHtml4((String)String.valueOf(this.packedSize)) + "</i>. Continue processing?")) {
                throw new CancelledException("Cancelled due to footer length field error.");
            }
        } else {
            this.payloadOriginalSize = -1L;
        }
        this.footerHashes = new LinkedHashMap<String, byte[]>();
        ArrayList<String> missingHashes = new ArrayList<String>();
        for (String hashName : CartV1Constants.EXPECTED_HASHES.keySet()) {
            if (optionalFooterData.has(hashName)) {
                try {
                    this.footerHashes.put(hashName, HexFormat.of().parseHex(optionalFooterData.get(hashName).getAsString()));
                    continue;
                }
                catch (IllegalArgumentException e) {
                    throw new CartInvalidCartException("Bad " + hashName + " hash format.");
                }
            }
            missingHashes.add(hashName);
        }
        if (this.footerHashes.isEmpty()) {
            if (!CartCancelDialogs.promptErrorContinue("No Hashes", "No hash data in CaRT footer metadata. Cannot verify content. Continue processing?")) {
                throw new CancelledException("Cancelled due to no hash data.");
            }
        } else if (!missingHashes.isEmpty() && !CartCancelDialogs.promptErrorContinue("Missing Hashes", "Expected hash(es) missing: " + String.join((CharSequence)", ", missingHashes) + ". Continue processing?")) {
            throw new CancelledException("Cancelled due to missing hash data (" + String.join((CharSequence)", ", missingHashes) + ").");
        }
    }

    private CartV1Decryptor createDecryptor() throws IOException, CartInvalidCartException, CartInvalidARC4KeyException, CartConfigurationException, CancelledException {
        if (this.header == null) {
            throw new CartInvalidCartException("CaRT header not initialized.");
        }
        try {
            return this.createDecryptor(this.header.arc4Key());
        }
        catch (CartInvalidARC4KeyException cartInvalidARC4KeyException) {
            ArrayList<byte[]> possibleKeys = new ArrayList<byte[]>();
            try {
                byte[] iniKey = this.getIniKey();
                if (iniKey != null) {
                    possibleKeys.add(iniKey);
                }
            }
            catch (IOException iOException) {
                // empty catch block
            }
            possibleKeys.add(CartV1Constants.DEFAULT_ARC4_KEY);
            possibleKeys.add(CartV1Constants.PRIVATE_ARC4_KEY_PLACEHOLDER);
            for (byte[] key : possibleKeys) {
                try {
                    return this.createDecryptor(key);
                }
                catch (CartInvalidARC4KeyException cartInvalidARC4KeyException2) {
                }
            }
            throw new CartInvalidARC4KeyException("Private CaRT ARC4 key could not be determined.");
        }
    }

    private CartV1Decryptor createDecryptor(String proposedArc4Key) throws IOException, CartInvalidCartException, CartInvalidARC4KeyException, CartConfigurationException, CancelledException {
        try {
            byte[] arc4Key = Arrays.copyOf(proposedArc4Key.getBytes(), 16);
            return this.createDecryptor(arc4Key);
        }
        catch (CartInvalidARC4KeyException arc4Key) {
            if (proposedArc4Key.length() >= 4 && proposedArc4Key.length() <= 20 && proposedArc4Key.length() % 4 == 0) {
                try {
                    byte[] b64key = Base64.getDecoder().decode(proposedArc4Key);
                    if (b64key.length != 16) {
                        b64key = Arrays.copyOf(b64key, 16);
                    }
                    return this.createDecryptor(b64key);
                }
                catch (CartInvalidARC4KeyException | IllegalArgumentException exception) {
                    // empty catch block
                }
            }
            throw new CartInvalidARC4KeyException("Private CaRT ARC4 key could not be determined.");
        }
    }

    private CartV1Decryptor createDecryptor(byte[] proposedArc4Key) throws IOException, CartInvalidCartException, CartInvalidARC4KeyException, CartConfigurationException, CancelledException {
        if (proposedArc4Key.length != 16) {
            proposedArc4Key = Arrays.copyOf(proposedArc4Key, 16);
        }
        if (this.testKey(proposedArc4Key)) {
            this.decryptor = new CartV1Decryptor(proposedArc4Key);
            return this.decryptor;
        }
        throw new CartInvalidARC4KeyException("Private CaRT ARC4 key could not be determined.");
    }

    private boolean testKey(byte[] potentialARC4key) throws IOException, CartInvalidCartException, CartInvalidARC4KeyException, CancelledException {
        return CartV1PayloadExtractor.testExtraction(this.internalReader, this, potentialARC4key);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private byte[] getIniKey() throws IOException, CartConfigurationException {
        String configFileName = System.getProperty("user.home") + "/.cart/cart.cfg";
        byte[] validKey = null;
        try (BufferedReader configReader = new BufferedReader(new FileReader(configFileName));){
            String line;
            Pattern pattern = Pattern.compile("^\\s*rc4_key\\s*=\\s*(([^\\s]{4}){1,5})\\s*(;.*)?$", 2);
            while ((line = configReader.readLine()) != null) {
                Matcher matcher = pattern.matcher(line);
                if (!matcher.find()) continue;
                try {
                    byte[] potentialARC4Key = Base64.getDecoder().decode(matcher.group(1));
                    validKey = Arrays.copyOf(potentialARC4Key, 16);
                    return validKey;
                }
                catch (IllegalArgumentException e) {
                    throw new CartConfigurationException("rc4_key in " + configFileName + " is not valid base64-encoded data.");
                    return validKey;
                }
            }
        }
    }

    public String getName() {
        return this.name;
    }

    public String getPath() {
        return this.path;
    }

    public long getDataOffset() {
        return this.dataOffset;
    }

    public long getDataSize() {
        return this.payloadOriginalSize;
    }

    public long getPackedSize() {
        return this.packedSize;
    }

    public byte[] getFooterHash(String hashName) {
        if (this.footerHashes.containsKey(hashName)) {
            return this.footerHashes.get(hashName);
        }
        return null;
    }

    public CartV1Header getHeader() {
        return this.header;
    }

    public CartV1Footer getFooter() {
        return this.footer;
    }

    public CartV1Decryptor getDecryptor() {
        return this.decryptor;
    }

    public JsonObject getMetadata() {
        JsonObject metadata = this.header.optionalHeaderData();
        if (metadata == null) {
            metadata = new JsonObject();
        }
        JsonObject optionalFooterData = null;
        try {
            optionalFooterData = this.footer.optionalFooterData();
        }
        catch (CartInvalidCartException cartInvalidCartException) {
            // empty catch block
        }
        if (optionalFooterData != null) {
            for (Map.Entry entry : optionalFooterData.entrySet()) {
                metadata.add((String)entry.getKey(), (JsonElement)entry.getValue());
            }
        }
        return metadata;
    }

    public static boolean isCart(ByteProvider byteProvider) {
        return CartV1File.isCart(new BinaryReader(byteProvider, true));
    }

    public static boolean isCart(BinaryReader reader) {
        return CartV1File.hasCartHeader(reader) && CartV1File.hasCartFooter(reader);
    }

    public static boolean hasCartHeader(ByteProvider byteProvider) {
        return CartV1File.hasCartHeader(new BinaryReader(byteProvider, true));
    }

    public static boolean hasCartHeader(BinaryReader reader) {
        try {
            CartV1Header header = new CartV1Header(reader);
            return true;
        }
        catch (CartInvalidCartException | IOException exception) {
            return false;
        }
    }

    public static boolean hasCartFooter(ByteProvider byteProvider) {
        return CartV1File.hasCartFooter(new BinaryReader(byteProvider, true));
    }

    public static boolean hasCartFooter(BinaryReader reader) {
        try {
            CartV1Footer footer = new CartV1Footer(reader);
            return true;
        }
        catch (CartInvalidCartException | IOException exception) {
            return false;
        }
    }
}

