﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/dynamodb/DynamoDB_EXPORTS.h>
#include <aws/dynamodb/model/AttributeValue.h>
#include <aws/dynamodb/model/ReturnValuesOnConditionCheckFailure.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace DynamoDB {
namespace Model {

/**
 * <p>Represents a request to perform a <code>DeleteItem</code>
 * operation.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/Delete">AWS API
 * Reference</a></p>
 */
class Delete {
 public:
  AWS_DYNAMODB_API Delete() = default;
  AWS_DYNAMODB_API Delete(Aws::Utils::Json::JsonView jsonValue);
  AWS_DYNAMODB_API Delete& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_DYNAMODB_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The primary key of the item to be deleted. Each element consists of an
   * attribute name and a value for that attribute.</p>
   */
  inline const Aws::Map<Aws::String, AttributeValue>& GetKey() const { return m_key; }
  inline bool KeyHasBeenSet() const { return m_keyHasBeenSet; }
  template <typename KeyT = Aws::Map<Aws::String, AttributeValue>>
  void SetKey(KeyT&& value) {
    m_keyHasBeenSet = true;
    m_key = std::forward<KeyT>(value);
  }
  template <typename KeyT = Aws::Map<Aws::String, AttributeValue>>
  Delete& WithKey(KeyT&& value) {
    SetKey(std::forward<KeyT>(value));
    return *this;
  }
  template <typename KeyKeyT = Aws::String, typename KeyValueT = AttributeValue>
  Delete& AddKey(KeyKeyT&& key, KeyValueT&& value) {
    m_keyHasBeenSet = true;
    m_key.emplace(std::forward<KeyKeyT>(key), std::forward<KeyValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Name of the table in which the item to be deleted resides. You can also
   * provide the Amazon Resource Name (ARN) of the table in this parameter.</p>
   */
  inline const Aws::String& GetTableName() const { return m_tableName; }
  inline bool TableNameHasBeenSet() const { return m_tableNameHasBeenSet; }
  template <typename TableNameT = Aws::String>
  void SetTableName(TableNameT&& value) {
    m_tableNameHasBeenSet = true;
    m_tableName = std::forward<TableNameT>(value);
  }
  template <typename TableNameT = Aws::String>
  Delete& WithTableName(TableNameT&& value) {
    SetTableName(std::forward<TableNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A condition that must be satisfied in order for a conditional delete to
   * succeed.</p>
   */
  inline const Aws::String& GetConditionExpression() const { return m_conditionExpression; }
  inline bool ConditionExpressionHasBeenSet() const { return m_conditionExpressionHasBeenSet; }
  template <typename ConditionExpressionT = Aws::String>
  void SetConditionExpression(ConditionExpressionT&& value) {
    m_conditionExpressionHasBeenSet = true;
    m_conditionExpression = std::forward<ConditionExpressionT>(value);
  }
  template <typename ConditionExpressionT = Aws::String>
  Delete& WithConditionExpression(ConditionExpressionT&& value) {
    SetConditionExpression(std::forward<ConditionExpressionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>One or more substitution tokens for attribute names in an expression.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetExpressionAttributeNames() const { return m_expressionAttributeNames; }
  inline bool ExpressionAttributeNamesHasBeenSet() const { return m_expressionAttributeNamesHasBeenSet; }
  template <typename ExpressionAttributeNamesT = Aws::Map<Aws::String, Aws::String>>
  void SetExpressionAttributeNames(ExpressionAttributeNamesT&& value) {
    m_expressionAttributeNamesHasBeenSet = true;
    m_expressionAttributeNames = std::forward<ExpressionAttributeNamesT>(value);
  }
  template <typename ExpressionAttributeNamesT = Aws::Map<Aws::String, Aws::String>>
  Delete& WithExpressionAttributeNames(ExpressionAttributeNamesT&& value) {
    SetExpressionAttributeNames(std::forward<ExpressionAttributeNamesT>(value));
    return *this;
  }
  template <typename ExpressionAttributeNamesKeyT = Aws::String, typename ExpressionAttributeNamesValueT = Aws::String>
  Delete& AddExpressionAttributeNames(ExpressionAttributeNamesKeyT&& key, ExpressionAttributeNamesValueT&& value) {
    m_expressionAttributeNamesHasBeenSet = true;
    m_expressionAttributeNames.emplace(std::forward<ExpressionAttributeNamesKeyT>(key),
                                       std::forward<ExpressionAttributeNamesValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>One or more values that can be substituted in an expression.</p>
   */
  inline const Aws::Map<Aws::String, AttributeValue>& GetExpressionAttributeValues() const { return m_expressionAttributeValues; }
  inline bool ExpressionAttributeValuesHasBeenSet() const { return m_expressionAttributeValuesHasBeenSet; }
  template <typename ExpressionAttributeValuesT = Aws::Map<Aws::String, AttributeValue>>
  void SetExpressionAttributeValues(ExpressionAttributeValuesT&& value) {
    m_expressionAttributeValuesHasBeenSet = true;
    m_expressionAttributeValues = std::forward<ExpressionAttributeValuesT>(value);
  }
  template <typename ExpressionAttributeValuesT = Aws::Map<Aws::String, AttributeValue>>
  Delete& WithExpressionAttributeValues(ExpressionAttributeValuesT&& value) {
    SetExpressionAttributeValues(std::forward<ExpressionAttributeValuesT>(value));
    return *this;
  }
  template <typename ExpressionAttributeValuesKeyT = Aws::String, typename ExpressionAttributeValuesValueT = AttributeValue>
  Delete& AddExpressionAttributeValues(ExpressionAttributeValuesKeyT&& key, ExpressionAttributeValuesValueT&& value) {
    m_expressionAttributeValuesHasBeenSet = true;
    m_expressionAttributeValues.emplace(std::forward<ExpressionAttributeValuesKeyT>(key),
                                        std::forward<ExpressionAttributeValuesValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Use <code>ReturnValuesOnConditionCheckFailure</code> to get the item
   * attributes if the <code>Delete</code> condition fails. For
   * <code>ReturnValuesOnConditionCheckFailure</code>, the valid values are: NONE and
   * ALL_OLD.</p>
   */
  inline ReturnValuesOnConditionCheckFailure GetReturnValuesOnConditionCheckFailure() const {
    return m_returnValuesOnConditionCheckFailure;
  }
  inline bool ReturnValuesOnConditionCheckFailureHasBeenSet() const { return m_returnValuesOnConditionCheckFailureHasBeenSet; }
  inline void SetReturnValuesOnConditionCheckFailure(ReturnValuesOnConditionCheckFailure value) {
    m_returnValuesOnConditionCheckFailureHasBeenSet = true;
    m_returnValuesOnConditionCheckFailure = value;
  }
  inline Delete& WithReturnValuesOnConditionCheckFailure(ReturnValuesOnConditionCheckFailure value) {
    SetReturnValuesOnConditionCheckFailure(value);
    return *this;
  }
  ///@}
 private:
  Aws::Map<Aws::String, AttributeValue> m_key;

  Aws::String m_tableName;

  Aws::String m_conditionExpression;

  Aws::Map<Aws::String, Aws::String> m_expressionAttributeNames;

  Aws::Map<Aws::String, AttributeValue> m_expressionAttributeValues;

  ReturnValuesOnConditionCheckFailure m_returnValuesOnConditionCheckFailure{ReturnValuesOnConditionCheckFailure::NOT_SET};
  bool m_keyHasBeenSet = false;
  bool m_tableNameHasBeenSet = false;
  bool m_conditionExpressionHasBeenSet = false;
  bool m_expressionAttributeNamesHasBeenSet = false;
  bool m_expressionAttributeValuesHasBeenSet = false;
  bool m_returnValuesOnConditionCheckFailureHasBeenSet = false;
};

}  // namespace Model
}  // namespace DynamoDB
}  // namespace Aws
